<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Appointment;
use App\Models\Document;
use App\Models\Rooms;
use App\Models\AppointmentReport;
use App\Models\Laboratory;
use App\Models\PatientOrders;
use App\Models\Pharmaceutical;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class HomeController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function index()
    {
        return view('admin.home');
    }
    public function adminHome()
    {
        $user = Auth::user();
        $recentAppAdmin = Appointment::with('user', 'provider')
        ->where('status', 'pending')
        ->whereDate('appointment_date', '>=', now())
        ->where('is_active', 1)
        ->where('is_deleted', 0)
        ->orderBy('created_at','desc')
        ->get();
        $recentAppProviders = Appointment::with('user', 'provider')
        ->where('provider_id', $user->id)
        ->where('status', 'pending')
        ->whereDate('appointment_date', '>=', now())
        ->where('is_active', 1)
        ->where('is_deleted', 0)
        ->orderBy('created_at','desc')
        ->get();
        $recentAppCustomer = Appointment::with('user', 'provider')
        ->where('user_id', $user->id)
        ->where('status', 'pending')
        ->whereDate('appointment_date', '>=', now())
        ->where('is_active', 1)
        ->where('is_deleted', 0)
        ->orderBy('created_at','desc')
        ->get();
        return view('admin.index', compact('recentAppAdmin', 'recentAppProviders','recentAppCustomer'))->with('title','Home');
    }

    public function profileEdit($id=null)
    {
        if($id == null){
            $id = Auth::user()->id;
        }
        $profile = User::find($id);
        return view('admin.pages.profile.profile', compact('profile'))->with('title', 'Manage Profile');
    }
    public function profileUpdate(Request $request)
    {
        $profile = User::find($request->user_id);

        if ($profile->profile_edit == 0) {
            $validated = Validator::make($request->all(), [
                'social_security_number' => 'required|unique:users|max:255'
            ]);

            if ($validated->fails()) {
                return redirect()->back()->with('validationerror', $validated->errors()->all());
            }
        }

        $extension = array("jpeg", "jpg", "png", "jfif");
        if (isset($request->image)) {
            $file = $request->image;
            $ext = $request->image->getClientOriginalExtension();
            if (in_array($ext, $extension)) {
                $file_name = $request->image->getClientOriginalName();
                $file_name = substr($file_name, 0, strpos($file_name, "."));
                $name = "uploads/profile/" . $file_name . "_" . time() . '.' . $file->getClientOriginalExtension();
                $destinationPath = public_path() . '/uploads/profile/';
                $share = $request->image->move($destinationPath, $name);
                $image = $name;
                $profile->image = $image;
            } else {
                $warning = "This File type is not Supported!";
                return redirect()->back()->with('error', "Error Code: " . $warning);
            }
        }

        $profile->name = $request->name;
        $profile->social_security_number = $request->social_security_number;
        $profile->phone = $request->phone;
        $profile->address = $request->address;
        $profile->date_of_birth = $request->date_of_birth;
        $profile->profile_edit = 1;
        // dd($profile);
        $profile->save();
        return redirect()->back()->with('message', 'Profile Updated Successfully');
    }

    public function PasswordEdit()
    {
        $passUpdate = User::find(Auth::user()->id);
        return view('admin.pages.updatepassword', compact('passUpdate'))->with('title', 'Update Password');
    }

    public function PasswordUppdate(Request $request)
    {
        $validated = Validator::make($request->all(), [
            'current_password' => 'required',
            'newpassword' => 'required|min:6',
            'cnfpassword' => 'required_with:newpassword|same:newpassword|min:6'
        ]);
        $auth = Auth::user();
        if (!Hash::check($request->get('current_password'), $auth->password)) {
            return back()->with('error', "Current Password is Invalid");
        }

        if (strcmp($request->get('current_password'), $request->newpassword) == 0) {
            return redirect()->back()->with("error", "New Password cannot be same as your current password.");
        }
        if ($validated->fails()) {
            return redirect()->back()->with('validationerror', $validated->errors()->all());
        }
        $password = Hash::make($request->newpassword);
        $passUpdate =  User::find($auth->id);
        $passUpdate->password = $password;
        $passUpdate->save();
        return redirect()->back()->with('message', 'Password Updated Successfully');
    }

    public function memberAppointmentDelete(Request $request)
    {
        if ($request->id) {
            $appointment = Appointment::where('id', $request->id)->first();
            if ($appointment) {
                $appointment->is_active = 0;
                $appointment->is_deleted = 1;
                $appointment->save();

                return response()->json(['status' => 1, 'message' => 'appointment Deleted Successfully!']);
            }
        }
        return response()->json(['status' => 0, 'message' => 'appointment Not Found!']);
    }

    public function patientreportsSave(Request $request)
    {
        $report = new Document();

        $validated = Validator::make($request->all(), [
            'file' => 'required|max:20480',
        ]);
        if ($validated->fails()) {
            return redirect()->back()->with('validationerror', $validated->errors()->all());
        }

        if ($request->hasFile('file')) {
            $file = $request->file('file');
            $file_name = $file->getClientOriginalName();
            $file_size = $file->getSize();
            if ($file_size > 20971520) {
                return redirect()->back()->with('error', 'Please upload a file less than 20MB.');
            }
            $file_name = pathinfo($file_name, PATHINFO_FILENAME);
            $file_extension = $file->getClientOriginalExtension();
            $file_name = $file_name . '_' . time() . '.' . $file_extension;
            $destinationPath = public_path('uploads/reports');
            $file->move($destinationPath, $file_name);
            $report->file = 'uploads/reports/' . $file_name;
            $report->file_name = $file->getClientOriginalName();
            $report->file_size = $file_size;
        }

        $report->user_id = $request->user_id ?? '';
        $report->upload_by = $request->upload_by ?? '';
        $report->save();
        return redirect()->back()->with('message', 'Report Uploaded Successfully');
    }
    public function patientreportsDelete(Request $request)
    {
        if ($request->id) {
            $report = Document::where('id', $request->id)->first();
            if ($report) {
                $report->is_active = 0;
                $report->is_deleted = 1;
                $report->save();

                return response()->json(['status' => 1, 'message' => 'Report Deleted Successfully!']);
            }
        }
        return response()->json(['status' => 0, 'message' => 'Report Not Found!']);
    }

    public function getRooms(Request $request)
    {
        $rooms = Rooms::where('provider_id', $request->provider_id)->get();
        $rooms->map(function($room) use($request) {
            $date = date("Y-m-d", strtotime($request->appointment_date." ".$room->day));
            $appointments = Appointment::where('provider_id', $room->provider_id)
            ->whereDate('appointment_date', $date)
            ->get()->pluck('appointment_time')->toArray();
            $room->booked_times = $appointments;
            $room->day_date = $date;
        });

        $data = [
            'rooms'=> $rooms
        ];
        return response()->json($data);
    }

    public function appointmentCheckIn(Request $request)
    {
        $appointment = Appointment::where('id', $request->id)->first();
        $appointment->user_id = $request->user_id;
        $appointment->check_in = date("Y-m-d H:i:s");
        $appointment->is_checked = 1;
        $appointment->status = 'completed';
        $appointment->save();
        return response()->json(['status' => 200,'success'=>true, 'appointment'=> $appointment, 'message' => 'Pateint CheckIn']);
    }

    public function appointmentCheckOut(Request $request)
    {
        $appointment = Appointment::where('id', $request->id)->first();
        $appointment->user_id = $request->user_id;
        $appointment->check_out = date("Y-m-d H:i:s") ?? '';
        $appointment->save();
        return response()->json(['status' => 200,'success'=>true, 'appointment'=> $appointment, 'message' => 'Pateint CheckOut']);
    }

    public function reScheduleAppointment(Request $request)
    {
        $appointment = Appointment::where('id', $request->id)->first();
        $appointment->user_id = $request->user_id;
        $appointment->appointment_date = $request->appointment_date ?? '';
        $appointment->appointment_time = $request->appointment_time ?? '';
        $appointment->save();

        $provider = User::find($request->provider_id);
        $user = User::find($request->user_id);

        $data = [
            'provider' => $provider->name ?? '',
            'name' => $user->name ?? '',
            'email' => $user->email ?? '',
            'number' => $user->phone ?? '',
            'appointmentdate' => $request->appointment_date ?? '',
            'appointmenttime' => $request->appointment_time ?? '',
            'roletype' => $provider->role_id ?? '',
        ];

        $providerMail = [$provider->email];
        $pateintMail = [$user->email];

        Mail::send('web.pages.mailing.memberAppointmentReschedule', $data, function ($message) use ($providerMail) {
            $message->to($providerMail);
            $message->subject('Appointment Schedule');
        });

        $data['roletype'] = $user->role_id ?? '';
        Mail::send('web.pages.mailing.memberAppointmentReschedule', $data, function ($message) use ($pateintMail) {
            $message->to($pateintMail);
            $message->subject('Appointment Schedule');
        });

        return response()->json(['status' => 200,'success'=>true, 'appointment'=> $appointment, 'message' => 'Appointment Re-schedule Successfully']);
    }

    public function markAsRead()
    {
        $user = auth()->user();
        $user->unreadNotifications->markAsRead();
        return redirect()->back();
    }

    public function laboratoryView(){
        $laboratory = Laboratory::where('is_active', 1)->where('is_deleted', 0)->get();
        return view('admin.pages.laboratory.laboratoryView', compact('laboratory'))->with('title','Laboratory view');
    }

    public function laboratoryCreate($id){
        if ($id > 0) {
            $laboratory = Laboratory::find($id);
            if ($laboratory) {
                return view('admin.pages.laboratory.laboratorySave', compact('laboratory'))->with('title', 'Laboratory edit');
            }
        }
        return view('admin.pages.laboratory.laboratorySave')->with('title','Laboratory create');
    }

    public function laboratorySave(Request $request){
        $user = Auth::user();
        $laboratoryId = $request->laboratory_id;

        if ($laboratoryId) {
            $laboratory = Laboratory::find($laboratoryId);
            if (!$laboratory) {
                return redirect()->back()->with('error', 'Laboratory not found.');
            }
        } else {
            $laboratory = new Laboratory();
        }

        $laboratory->user_id = $user->id ?? 0;
        $laboratory->name = $request->name ?? '';
        $laboratory->email = $request->email ?? '';
        $laboratory->phone = $request->phone ?? '';
        $laboratory->address = $request->address ?? '';
        $laboratory->save();
        return redirect()->back()->with('message','Record save successfully');
    }

    public function laboratoryDelete(Request $request)
    {
        if ($request->id) {
            $laboratory = Laboratory::where('id', $request->id)->first();
            if ($laboratory) {
                $laboratory->is_active = 0;
                $laboratory->is_deleted = 1;
                $laboratory->save();

                return response()->json(['status' => 1, 'message' => 'Package Deleted Successfully!']);
            }
        }
        return response()->json(['status' => 0, 'message' => 'Laboratory Not Found!']);
    }

    public function pharmaceuticalView(){
        $pharmaceutical = Pharmaceutical::where('is_active', 1)->where('is_deleted', 0)->get();
        return view('admin.pages.pharmaceutical.pharmaceuticalView', compact('pharmaceutical'))->with('title','pharmaceutical view');
    }

    public function pharmaceuticalCreate($id){
        if ($id > 0) {
            $pharmaceutical = Pharmaceutical::find($id);
            if ($pharmaceutical) {
                return view('admin.pages.pharmaceutical.pharmaceuticalSave', compact('pharmaceutical'))->with('title', 'pharmaceutical edit');
            }
        }
        return view('admin.pages.pharmaceutical.pharmaceuticalSave')->with('title','pharmaceutical create');
    }

    public function pharmaceuticalSave(Request $request){
        $user = Auth::user();
        $pharmaceuticalId = $request->pharmaceutical_id;

        if ($pharmaceuticalId) {
            $pharmaceutical = Pharmaceutical::find($pharmaceuticalId);
            if (!$pharmaceutical) {
                return redirect()->back()->with('error', 'pharmaceutical not found.');
            }
        } else {
            $pharmaceutical = new Pharmaceutical();
        }

        $pharmaceutical->user_id = $user->id ?? 0;
        $pharmaceutical->name = $request->name ?? '';
        $pharmaceutical->email = $request->email ?? '';
        $pharmaceutical->phone = $request->phone ?? '';
        $pharmaceutical->address = $request->address ?? '';
        $pharmaceutical->save();
        return redirect()->back()->with('message','Record save successfully');
    }

    public function pharmaceuticalDelete(Request $request)
    {
        if ($request->id) {
            $pharmaceutical = Pharmaceutical::where('id', $request->id)->first();
            if ($pharmaceutical) {
                $pharmaceutical->is_active = 0;
                $pharmaceutical->is_deleted = 1;
                $pharmaceutical->save();

                return response()->json(['status' => 1, 'message' => 'Package Deleted Successfully!']);
            }
        }
        return response()->json(['status' => 0, 'message' => 'pharmaceutical Not Found!']);
    }
    public function patientOrderView($id){
        $order = PatientOrders::where('patient_id', $id)->where('is_active', 1)->where('is_deleted', 0)->get();
        return view('admin.pages.patient.patientOrder.orderView', compact('order','id'))->with('title','patient charge history');
    }

    public function patientOrderCreate($id, $patient){
        $patient = User::find($patient);
        if ($id > 0) {
            $order = PatientOrders::find($id);
            if ($order) {
                return view('admin.pages.patient.patientOrder.orderSave', compact('order','patient'))->with('title', 'patient charge history edit');
            }
        }
        return view('admin.pages.patient.patientOrder.orderSave', compact('patient'))->with('title','patient charge history create');
    }

    public function patientOrderSave(Request $request){
        $orderId = $request->input('order_id');

        if ($orderId) {
            $order = PatientOrders::find($orderId);
            if (!$order) {
                return redirect()->back()->with('error', 'charge history not found.');
            }
        } else {
            $order = new PatientOrders();
        }

        $order->patient_id = $request->patient_id ??'';
        $order->amount = $request->amount ??'';
        $order->description = $request->description ??'';
        $order->save();
        return redirect()->back()->with('message','charge history save successfully');
    }

    public function patientOrderDelete(Request $request)
    {
        if ($request->id) {
            $order = PatientOrders::where('id', $request->id)->first();
            if ($order) {
                $order->is_active = 0;
                $order->is_deleted = 1;
                $order->save();

                return response()->json(['status' => 1, 'message' => 'charge history Deleted Successfully!']);
            }
        }
        return response()->json(['status' => 0, 'message' => 'charge history Not Found!']);
    }
}
